<?php
header('Content-Type: application/json');

$config = require __DIR__ . '/mail-config.php';

/**
 * Debug logger
 */
function logDebug($label, $data = null)
{
    $logFile = __DIR__ . '/mail-debug.log';
    $timestamp = date('c');

    $entry = "[{$timestamp}] {$label}";
    if ($data !== null) {
        $entry .= "\n" . (is_array($data) || is_object($data)
            ? print_r($data, true)
            : $data);
    }

    $entry .= "\n" . str_repeat('-', 80) . "\n";
    file_put_contents($logFile, $entry, FILE_APPEND | LOCK_EX);
}

/**
 * Basic error logger
 */
function logError($message)
{
    $logFile = __DIR__ . '/error.log';
    $timestamp = date('c');
    file_put_contents(
        $logFile,
        "[{$timestamp}] {$message}\n",
        FILE_APPEND | LOCK_EX
    );
}

/**
 * PHPMailer bootstrap
 */
$phpMailerBase = __DIR__ . '/vendor/PHPMailer/src/';
if (!file_exists($phpMailerBase . 'PHPMailer.php')) {
    logError('PHPMailer missing');
    http_response_code(500);
    echo json_encode(['status' => 'error']);
    exit;
}

require_once $phpMailerBase . 'PHPMailer.php';
require_once $phpMailerBase . 'SMTP.php';
require_once $phpMailerBase . 'Exception.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

try {
    /**
     * Validate required fields
     */
    $requiredFields = [
        'name',
        'organization',
        'email',
        'phone',
        'incident_type',
        'urgency',
        'message'
    ];

    foreach ($requiredFields as $field) {
        if (empty($_POST[$field])) {
            throw new Exception("Missing required field: {$field}");
        }
    }

    /**
     * Sanitize input
     */
    $name = trim(strip_tags($_POST['name']));
    $organization = trim(strip_tags($_POST['organization']));
    $email = filter_var($_POST['email'], FILTER_SANITIZE_EMAIL);
    $phone = trim(strip_tags($_POST['phone']));
    $incidentType = trim(strip_tags($_POST['incident_type']));
    $urgency = trim(strip_tags($_POST['urgency']));
    $message = trim(strip_tags($_POST['message']));

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Invalid email address');
    }

    /**
     * Log incoming request (privacy-safe)
     */
    logDebug('Incoming contact form', [
        'name' => $name,
        'organization' => $organization,
        'email' => $email,
        'phone' => $phone,
        'incident_type' => $incidentType,
        'urgency' => $urgency,
        'message_length' => strlen($message),
    ]);

    /**
     * Log SMTP config (NO password)
     */
    logDebug('SMTP config', [
        'host' => $config['host'],
        'port' => $config['port'],
        'encryption' => $config['encryption'],
        'username' => $config['username'],
        'from_email' => $config['from_email'],
        'to_email' => $config['to_email'],
    ]);

    /**
     * ============================
     * ADMIN NOTIFICATION MAIL
     * ============================
     */
    $mailer = new PHPMailer(true);
    $mailer->isSMTP();
    $mailer->Host = $config['host'];
    $mailer->SMTPAuth = true;
    $mailer->Username = $config['username'];
    $mailer->Password = $config['password'];
    $mailer->SMTPSecure = $config['encryption'];
    $mailer->Port = $config['port'];

    $mailer->CharSet = 'UTF-8';
    $mailer->isHTML(false);

    // IMPORTANT: keep sender consistent with SMTP domain
    $mailer->setFrom($config['from_email'], $config['from_name']);
    $mailer->Sender = $config['from_email'];

    $mailer->addAddress($config['to_email'], $config['to_name']);

    // Do NOT reply directly to user email (avoids SMTP rejection)
    $mailer->addReplyTo($config['from_email'], $config['from_name']);
    $mailer->addCustomHeader('X-User-Email', $email);

    $mailer->Subject = 'New Incident Report - Cyber Certified Line';
    $mailer->Body =
        "Full name: {$name}\n" .
        "Agency/Organization: {$organization}\n" .
        "Email: {$email}\n" .
        "Phone: {$phone}\n" .
        "Incident Type: {$incidentType}\n" .
        "Urgency: {$urgency}\n\n" .
        "Incident Summary:\n{$message}\n";

    // SMTP debug capture
    $mailer->SMTPDebug = 2;
    $mailer->Debugoutput = function ($str, $level) {
        logDebug("SMTP DEBUG [{$level}]", $str);
    };

    logDebug('Sending admin email', [
        'from' => $config['from_email'],
        'to' => $config['to_email'],
        'subject' => $mailer->Subject,
    ]);

    $mailer->send();

    /**
     * ============================
     * USER CONFIRMATION MAIL
     * ============================
     */
    $confirmationMailer = new PHPMailer(true);
    $confirmationMailer->isSMTP();
    $confirmationMailer->Host = $config['host'];
    $confirmationMailer->SMTPAuth = true;
    $confirmationMailer->Username = $config['username'];
    $confirmationMailer->Password = $config['password'];
    $confirmationMailer->SMTPSecure = $config['encryption'];
    $confirmationMailer->Port = $config['port'];

    $confirmationMailer->CharSet = 'UTF-8';
    $confirmationMailer->isHTML(true);

    $confirmationMailer->setFrom($config['from_email'], $config['from_name']);
    $confirmationMailer->Sender = $config['from_email'];
    $confirmationMailer->addAddress($email, $name);

    $confirmationMailer->Subject =
        'We received your request - Cyber Certified Line';

    $telegramLink = 'https://t.me/fedralbureauof_investigation';

    $confirmationMailer->Body = "
        <div style=\"font-family: Arial, sans-serif; background:#f5f7fb; padding:24px;\">
            <div style=\"max-width:640px;margin:auto;background:#fff;padding:32px;border-radius:12px;\">
                <h2>We received your request</h2>
                <p>Hello {$name},</p>
                <p>
                    Thank you for contacting Cyber Certified Line.
                    Your report is currently being reviewed by our incident response team.
                </p>
                <p>
                    For the fastest response, reach us on Telegram:
                    <a href=\"{$telegramLink}\">@cybercertifiedline</a>
                </p>
                <p style=\"margin-top:24px;color:#6b7280;font-size:14px;\">
                    — Cyber Certified Line Intake Team
                </p>
            </div>
        </div>
    ";

    $confirmationMailer->AltBody =
        "Hello {$name},\n\n" .
        "We have received your report and our team is reviewing it.\n\n" .
        "Telegram: {$telegramLink}\n\n" .
        "— Cyber Certified Line Intake Team";

    logDebug('Sending confirmation email', [
        'to' => $email,
        'subject' => $confirmationMailer->Subject,
    ]);

    $confirmationMailer->send();

    echo json_encode([
        'status' => 'ok',
        'message' => 'Your report has been submitted successfully.'
    ]);
} catch (Exception $e) {
    logError('Contact form error: ' . $e->getMessage());
    logDebug('Exception trace', $e->getTraceAsString());

    http_response_code(500);
    echo json_encode([
        'status' => 'error',
        'message' => 'Unable to submit your report at this time.'
    ]);
}
